<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Movie Display with Details</title>
    <style>
        body {
            margin: 0;
            padding: 0;
            position: relative;
            height: 100vh;
            background-color: #222;
            font-family: Arial, sans-serif;
        }
        .backdrop-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: -1;
        }
        .backdrop {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        .main-container {
            position: absolute;
            top: 0;
            left: 0;
            padding: 25px;
            width: 350px;
            background: linear-gradient(to right, rgba(0,0,0,0.8) 70%, transparent);
            height: 100%;
            display: flex;
            flex-direction: column;
            z-index: 2;
        }
        .logo-section {
            margin-bottom: 25px;
        }
        .logo {
            max-width: 250px;
            margin-bottom: 15px;
        }
        .movie-stats {
            color: white;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.3);
            font-size: 0.9em;
        }
        .rating-container {
            margin-bottom: 10px;
        }
        .trending-badge {
            display: inline-block;
            background: rgba(255,215,0,0.9);
            color: black;
            padding: 4px 10px;
            border-radius: 12px;
            font-weight: bold;
            font-size: 0.75em;
            margin-bottom: 2px;
        }
        .rating {
            font-size: 1.6em;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .star {
            color: #ffd700;
        }
        .genres {
            margin: 10px 0;
            font-size: 0.95em;
            color: #e1e1e1;
        }
        .additional-info {
            display: flex;
            flex-direction: column;
            gap: 8px;
            margin-top: 15px;
            font-size: 0.9em;
        }
        .info-item {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .icon {
            width: 20px;
            text-align: center;
            font-size: 0.9em;
        }
        .divider {
            height: 1px;
            background: rgba(255,255,255,0.2);
            margin: 20px 0;
            width: 100%;
        }
    </style>
</head>
<body>
    <div class="backdrop-container">
        <img id="backdrop" class="backdrop">
    </div>
    
    <div class="main-container">
        <div class="logo-section">
            <img id="logo" class="logo">
        </div>
        
        <div class="movie-stats" id="movieStats">
            <div class="rating-container">
                <div class="trending-badge" id="trending"></div>
                <div class="rating">
                    <span class="star">★</span>
                    <span id="rating">0.0/10</span>
                </div>
            </div>
            
            <div class="divider"></div>
            
            <div class="genres" id="genres">
                Géneros
            </div>
            
            <div class="additional-info">
                <div class="info-item">
                    <span class="icon">⏱</span>
                    <span id="runtime">Duración</span>
                </div>
                <div class="info-item">
                    <span class="icon">📅</span>
                    <span id="releaseYear">Año</span>
                </div>
                <div class="info-item">
                    <span class="icon">🌐</span>
                    <span id="language">Idioma</span>
                </div>
                <div class="info-item">
                    <span class="icon">👥</span>
                    <span id="voteCount">Votos</span>
                </div>
            </div>
        </div>
    </div>

    <script>
        const apiKey = '6b8e3eaa1a03ebb45642e9531d8a76d2';
        let currentIndex = 0;
        let movieIds = [];

        async function fetchPopularEnglishMovieIds() {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/discover/movie?api_key=${apiKey}&sort_by=popularity.desc&language=en`);
                if (!response.ok) throw new Error('Failed to fetch popular English movies');
                const data = await response.json();
                movieIds = data.results.map(movie => movie.id);
            } catch (error) {
                console.error('Error fetching popular English movies:', error);
            }
        }

        async function fetchMovieBackdrop(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}/images?api_key=${apiKey}`);
                if (!response.ok) throw new Error('Failed to fetch movie backdrop');
                const data = await response.json();
                const backdropPath = data.backdrops[0]?.file_path;
                return backdropPath ? `https://image.tmdb.org/t/p/original${backdropPath}` : null;
            } catch (error) {
                console.error('Error fetching movie backdrop:', error);
            }
        }

        async function fetchMovieLogo(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}?api_key=${apiKey}&append_to_response=images`);
                if (!response.ok) throw new Error('Failed to fetch movie logo');
                const data = await response.json();
                const logos = data.images.logos;
                const englishLogo = logos.find(logo => logo.iso_639_1 === 'en');
                return englishLogo ? `https://image.tmdb.org/t/p/w500${englishLogo.file_path}` : null;
            } catch (error) {
                console.error('Error fetching movie logo:', error);
            }
        }

        async function fetchMovieDetails(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}?api_key=${apiKey}&language=en`);
                if (!response.ok) throw new Error('Failed to fetch movie details');
                return await response.json();
            } catch (error) {
                console.error('Error fetching movie details:', error);
            }
        }

        function formatRuntime(minutes) {
            const hours = Math.floor(minutes / 60);
            const remainingMinutes = minutes % 60;
            return `${hours}h ${remainingMinutes}m`;
        }

        function formatNumber(number) {
            return new Intl.NumberFormat().format(number);
        }

        async function updateMovieDisplay() {
            if (movieIds.length === 0) {
                console.error('No movie IDs available.');
                return;
            }

            const movieId = movieIds[currentIndex];

            try {
                const [backdropUrl, logoUrl, movieDetails] = await Promise.all([
                    fetchMovieBackdrop(movieId),
                    fetchMovieLogo(movieId),
                    fetchMovieDetails(movieId)
                ]);

                if (backdropUrl) {
                    document.getElementById('backdrop').src = backdropUrl;
                }

                if (logoUrl) {
                    document.getElementById('logo').src = logoUrl;
                }

                if (movieDetails) {
                    // Actualizar rating
                    document.getElementById('rating').textContent = `${movieDetails.vote_average.toFixed(1)}/10`;
                    
                    // Actualizar géneros
                    document.getElementById('genres').textContent = 
                        movieDetails.genres.map(genre => genre.name).join(' • ');
                    
                    // Actualizar tiempo de duración
                    document.getElementById('runtime').textContent = 
                        formatRuntime(movieDetails.runtime);
                    
                    // Actualizar año
                    document.getElementById('releaseYear').textContent = 
                        new Date(movieDetails.release_date).getFullYear();
                    
                    // Actualizar idioma
                    document.getElementById('language').textContent = 
                        movieDetails.original_language.toUpperCase();
                    
                    // Actualizar conteo de votos
                    document.getElementById('voteCount').textContent = 
                        `${formatNumber(movieDetails.vote_count)} votos`;
                    
                    // Actualizar trending badge
                    document.getElementById('trending').textContent = 
                        `#${currentIndex + 1} Trending`;
                }

                currentIndex = (currentIndex + 1) % movieIds.length;
            } catch (error) {
                console.error('Error updating movie display:', error);
            }
        }

        fetchPopularEnglishMovieIds().then(() => {
            setInterval(updateMovieDisplay, 4000);
            updateMovieDisplay();
        });
    </script>
</body>
</html>