<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Movie Showcase - Proportional Design</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --neon-primary: #ff0000;
            --neon-secondary: #00ffff;
            --neon-text: #ffffff;
            --neon-shadow: rgba(255, 0, 0, 0.7);
        }

        body {
            background-color: #0a0a0a;
            font-family: 'Arial', sans-serif;
            color: var(--neon-text);
            height: 100vh;
            width: 100vw;
            overflow: hidden;
        }

        .main-container {
            position: relative;
            width: 100vw;
            height: 100vh;
            background: linear-gradient(135deg, rgba(0,0,0,0.3) 0%, rgba(0,0,0,0.7) 100%);
            display: flex;
            justify-content: center;
            align-items: flex-start; /* Cambiado de center a flex-start */
            padding-top: 1vh; /* Añadido padding-top para bajar un poco del borde superior */
        }

        .backdrop {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            object-fit: cover;
            z-index: -1;
        }

        .movie-panel {
            width: 70%;
            height: 70%;
            background: rgba(0, 0, 0, 0.8);
            border: 1px solid var(--neon-primary);
            border-radius: 10px;
            box-shadow: 0 0 20px var(--neon-shadow);
            display: flex;
            flex-direction: column;
            padding: 2% 2%;
            gap: 2vh;
        }

        .logo-section {
            height: 15%;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .movie-logo {
            height: 100%;
            max-width: 100%;
            object-fit: contain;
            filter: drop-shadow(0 0 10px var(--neon-shadow));
        }

        .gallery {
            height: 30%;
            display: flex;
            gap: 2%;
            justify-content: center;
            align-items: center;
            padding: 1%;
            background: rgba(0, 0, 0, 0.8);
            border: 1px solid var(--neon-primary);
            border-radius: 10px;
        }

        .gallery-image {
            height: 100%;
            width: 23%;
            object-fit: cover;
            border-radius: 8px;
            border: 2px solid var(--neon-primary);
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .gallery-image:hover {
            transform: scale(1.05);
            box-shadow: 0 0 20px var(--neon-shadow);
        }

        .movie-info {
            height: 15%;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            background: rgba(0, 0, 0, 0.6);
            border: 1px solid var(--neon-primary);
            border-radius: 10px;
        }

        .rating-value {
            font-size: 2em;
            color: var(--neon-secondary);
            text-shadow: 0 0 10px var(--neon-secondary);
        }

        .movie-details {
            display: flex;
            gap: 3vw;
            font-size: 1.0em;
        }

        .detail-item {
            color: var(--neon-text);
            text-shadow: 0 0 5px var(--neon-shadow);
        }

        .cast-grid {
            height: 30%;
            display: flex;
            gap: 2%;
            justify-content: center;
            align-items: center;
            padding: 1%;
        }

        .cast-card {
            height: 100%;
            width: calc(94% / 6); /* 94% dividido entre 6 cards, dejando 6% para gaps */
            padding: 1%;
            text-align: center;
            transition: transform 0.3s ease;
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        
        .cast-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 0 15px var(--neon-shadow);
        }
        
        .cast-text-container {
            width: 100%;
            display: flex;
            flex-direction: column;
            align-items: center;
        }

        .cast-image {
            width: 60%;
            aspect-ratio: 1/1;
            border-radius: 50%;
            margin-bottom: 5%;
            border: 2px solid var(--neon-primary);
            object-fit: cover;
        }

        .cast-name {
            font-size: 0.85em;
            margin-bottom: 3%;
            color: var(--neon-text);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            width: 100%;
            padding: 0 5px;
        }

        .cast-character {
            font-size: 0.7em;
            color: var(--neon-secondary);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            width: 100%;
            padding: 0 5px;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.9);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            max-width: 90%;
            max-height: 90vh;
            object-fit: contain;
        }

        .modal-close {
            position: absolute;
            top: 20px;
            right: 20px;
            color: white;
            font-size: 30px;
            cursor: pointer;
            background: none;
            border: none;
            text-shadow: 0 0 10px var(--neon-primary);
        }

        .navigation-dots {
            position: fixed;
            right: 2vw;
            top: 50%;
            transform: translateY(-50%);
            display: flex;
            flex-direction: column;
            gap: 1vh;
            z-index: 100;
        }

        .navigation-dots button {
            width: 1vw;
            height: 1vw;
            max-width: 12px;
            max-height: 12px;
            min-width: 6px;
            min-height: 6px;
            border-radius: 50%;
            border: none;
            background: rgba(255, 255, 255, 0.3);
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .navigation-dots button.active {
            background: var(--neon-primary);
            box-shadow: 0 0 10px var(--neon-primary);
        }

        @keyframes neonPulse {
            0% { opacity: 0.8; }
            50% { opacity: 1; }
            100% { opacity: 0.8; }
        }

        /* Ajustes para mantener la legibilidad en pantallas muy pequeñas */
        @media (max-height: 400px), (max-width: 400px) {
            .movie-details {
                font-size: 0.8em;
            }
            .cast-name {
                font-size: 0.8em;
            }
            .cast-character {
                font-size: 0.6em;
            }
            .rating-value {
                font-size: 2em;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <img id="backdrop" class="backdrop" alt="Movie Backdrop">
        
        <div class="movie-panel">
            <div class="logo-section">
                <img id="movieLogo" class="movie-logo" alt="Movie Logo">
            </div>

            <div class="gallery" id="imageGallery"></div>

            <div class="movie-info">
                <div class="rating-value">
                    <span id="rating">0.0</span>/10
                </div>
                <div class="movie-details">
                    <div class="detail-item">
                        <span id="runtime">0 min</span>
                    </div>
                    <div class="detail-item">
                        <span id="releaseYear">2024</span>
                    </div>
                    <div class="detail-item">
                        <span id="language">EN</span>
                    </div>
                </div>
            </div>

            <div class="cast-grid" id="castGrid"></div>
        </div>
    </div>

    <div id="imageModal" class="modal">
        <button class="modal-close">&times;</button>
        <img class="modal-content" id="modalImage">
    </div>

    <div id="navigationDots" class="navigation-dots"></div>

    <script>
        const apiKey = '6b8e3eaa1a03ebb45642e9531d8a76d2';
        let currentIndex = 0;
        let movieIds = [];
        let autoPlayInterval;

        async function fetchPopularMovies() {
            try {
                const response = await fetch(
                    `https://api.themoviedb.org/3/movie/popular?api_key=${apiKey}&language=en-US`
                );
                const data = await response.json();
                movieIds = data.results.map(movie => movie.id);
                setupNavigation();
                return data.results;
            } catch (error) {
                console.error('Error fetching popular movies:', error);
                return [];
            }
        }

        async function fetchMovieDetails(movieId) {
            try {
                const [details, credits] = await Promise.all([
                    fetch(`https://api.themoviedb.org/3/movie/${movieId}?api_key=${apiKey}&append_to_response=images`)
                        .then(res => res.json()),
                    fetch(`https://api.themoviedb.org/3/movie/${movieId}/credits?api_key=${apiKey}`)
                        .then(res => res.json())
                ]);
                return { details, credits };
            } catch (error) {
                console.error('Error fetching movie details:', error);
                return null;
            }
        }

        async function fetchMovieImages(movieId) {
            try {
                const response = await fetch(
                    `https://api.themoviedb.org/3/movie/${movieId}/images?api_key=${apiKey}`
                );
                const data = await response.json();
                return data.backdrops.slice(0, 4);
            } catch (error) {
                console.error('Error fetching movie images:', error);
                return [];
            }
        }

        function setupNavigation() {
            const nav = document.getElementById('navigationDots');
            nav.innerHTML = '';
            
            movieIds.forEach((_, index) => {
                const button = document.createElement('button');
                button.className = index === 0 ? 'active' : '';
                button.addEventListener('click', () => {
                    currentIndex = index;
                    updateMovie();
                    updateNavigationDots();
                });
                nav.appendChild(button);
            });
        }

        function updateNavigationDots() {
            const dots = document.querySelectorAll('.navigation-dots button');
            dots.forEach((dot, index) => {
                dot.className = index === currentIndex ? 'active' : '';
            });
        }

        function updateCastGrid(cast) {
            const grid = document.getElementById('castGrid');
            grid.innerHTML = '';

            cast.slice(0, 6).forEach(member => {
                if (!member.profile_path) return;

                const card = document.createElement('div');
                card.className = 'cast-card';
                
                card.innerHTML = `
                    <img class="cast-image" 
                         src="https://image.tmdb.org/t/p/w185${member.profile_path}" 
                         alt="${member.name}">
                    <div class="cast-text-container">
                        <div class="cast-name" title="${member.name}">${member.name}</div>
                        <div class="cast-character" title="${member.character}">${member.character}</div>
                    </div>
                `;
                
                grid.appendChild(card);
            });
        }

        function updateGallery(images) {
            const gallery = document.getElementById('imageGallery');
            gallery.innerHTML = images
                .map(image => `
                    <img class="gallery-image" 
                         src="https://image.tmdb.org/t/p/w500${image.file_path}" 
                         alt="Movie Image"
                         onclick="openModal(this.src)">
                `).join('');
        }

        function openModal(imageSrc) {
            const modal = document.getElementById('imageModal');
            const modalImg = document.getElementById('modalImage');
            modal.style.display = 'flex';
            modalImg.src = imageSrc.replace('/w500/', '/original/');
        }

        document.querySelector('.modal-close').onclick = function() {
            document.getElementById('imageModal').style.display = 'none';
        }

        window.onclick = function(event) {
            const modal = document.getElementById('imageModal');
            if (event.target === modal) {
                modal.style.display = 'none';
            }
        }

        async function updateMovie() {
            if (movieIds.length === 0) return;

            const movieId = movieIds[currentIndex];
            const [movieData, images] = await Promise.all([
                fetchMovieDetails(movieId),
                fetchMovieImages(movieId)
            ]);
            
            if (!movieData) return;

            const { details, credits } = movieData;

            // Actualizar backdrop
            document.getElementById('backdrop').src = 
                `https://image.tmdb.org/t/p/original${details.backdrop_path}`;

            // Actualizar logo
            const logo = details.images.logos.find(logo => logo.iso_639_1 === 'en');
            document.getElementById('movieLogo').src = logo
                ? `https://image.tmdb.org/t/p/w500${logo.file_path}`
                : `https://image.tmdb.org/t/p/w500${details.poster_path}`;

            // Actualizar información básica
            document.getElementById('rating').textContent = details.vote_average.toFixed(1);
            document.getElementById('runtime').textContent = `${details.runtime} min`;
            document.getElementById('releaseYear').textContent = 
                new Date(details.release_date).getFullYear();
            document.getElementById('language').textContent = 
                details.original_language.toUpperCase();

            // Actualizar galería y reparto
            updateGallery(images);
            updateCastGrid(credits.cast);
            updateNavigationDots();
        }

        function startAutoPlay() {
            stopAutoPlay();
            autoPlayInterval = setInterval(() => {
                currentIndex = (currentIndex + 1) % movieIds.length;
                updateMovie();
            }, 8000);
        }

        function stopAutoPlay() {
            if (autoPlayInterval) {
                clearInterval(autoPlayInterval);
            }
        }

        // Initialize
        async function initialize() {
            await fetchPopularMovies();
            await updateMovie();
            startAutoPlay();
        }

        initialize();
    </script>
</body>
</html>